<?php

/**
 * @file
 * Installation profile code.
 */

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Database\Database;
use Drupal\Core\Database\SchemaObjectExistsException;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Implements hook_form_FORM_ID_alter().
 *
 * Allows the profile to alter the site configuration form.
 */
function fast_form_install_configure_form_alter(&$form, FormStateInterface $form_state) {
  // Pre-populate the site name with the server name.
  $form['site_information']['site_name']['#default_value'] = $_SERVER['SERVER_NAME'];
  $form['#submit'][] = '_fast_form_submit';
}

/**
 * Form submit callback.
 */
function _fast_form_submit(array &$form, FormStateInterface &$form_state) {
  // Import database dump file.
  $dump_file = dirname(__FILE__) . '/dump.sql';
  $success = _fast_import($dump_file);

  if (!$success) {
    return;
  }

  // Now re-set the values they filled in during the previous step.
  /** @var \Drupal\Core\Config\ConfigFactoryInterface $config_factory */
  $config_factory = \Drupal::service('config.factory');
  $config = $config_factory->getEditable('system.site');
  $config->set('name', $form_state->getValue('site_name'));
  $config->set('mail', $form_state->getValue('site_mail'));
  $config->set('page', [
    '403' => '',
    '404' => '',
    'front' => '/node/1',
  ]);
  $config->save(TRUE);

  $config = $config_factory->getEditable('system.date');
  $config->set('timezone.default', $form_state->getValue('date_default_timezone'))->save(TRUE);

  // Perform additional clean-up tasks.
  $config = $config_factory->getEditable('system.file');
  $config->clear('path.temporary')->save(TRUE);

  // Replace their username and password and log them in.
  $account = $form_state->getValue('account');
  $connection = Database::getConnection();
  $connection->update('users_field_data')
    ->fields([
      'name' => $account['name'],
      'pass' => _fast_hash_password($account['pass']),
      'mail' => $account['mail'],
    ])
    ->condition('uid', 1)
    ->execute();

  $user = User::load(1);
  user_login_finalize($user);

  _fast_copy_directory('files', 'sites/default/files');

  $response = new RedirectResponse('/');
  $request  = \Drupal::request();
  $request->getSession()->save();
  $response->prepare($request);
  \Drupal::service('kernel')->terminate($request, $response);
  $response->send();
  die();
}

/**
 * Copy files to sites/default/files.
 */
function _fast_copy_directory($src, $dst) {
  if (!is_dir($src)) {
    return;
  }

  if (!is_dir($dst)) {
    mkdir($dst, 0777, true);
  }

  $dir = opendir($src);

  while (($file = readdir($dir)) !== false) {
    if ($file == '.' || $file == '..') {
      continue;
    }

    $src_path = $src . '/' . $file;
    $dst_path = $dst . '/' . $file;

    if (is_dir($src_path)) {
      _fast_copy_directory($src_path, $dst_path);
    }
    else {
      copy($src_path, $dst_path);
    }
  }

  closedir($dir);
}

/**
 * Hash a user's password in Drupal.
 *
 * @param string $password
 *   The plaintext password to hash.
 *
 * @return string
 *   The hashed password.
 */
function _fast_hash_password($password) {
  /** @var \Drupal\Core\Password\PasswordInterface $password_manager */
  $password_manager = \Drupal::service('password');
  return $password_manager->hash($password);
}

/**
 * Imports a database seven file.
 *
 * @see demo_reset().
 */
function _fast_import($filename) {
  if (!file_exists($filename) || !($file = fopen($filename, 'r'))) {
    \Drupal::messenger()->addError(t('Unable to open seven file %filename.', ['%filename' => $filename]));
    return FALSE;
  }

  _fast_drop_all_tables();
  $num = 0;
  $connection = _fast_get_mysqli_connection();

  while ($line = _fast_read_sql_command($file)) {
    if ($line) {
      // Execute the SQL query from the file.
      if (!$connection->query($line)) {
        // Log the error or handle it appropriately.
        \Drupal::messenger()->addError(t('Error executing query: %error', ['%error' => $connection->error]));
        return FALSE;
      }
      $num++;
    }
  }

  fclose($file);

  return $num;
}

/**
 * Get line from file.
 */
function _fast_read_sql_command($file) {
  $out = '';
  while ($line = fgets($file)) {
    $first2 = substr($line, 0, 2);
    $first3 = substr($line, 0, 2);

    // Ignore single line comments. This function doesn't support multiline
    // comments or inline comments.
    if ($first2 != '--' && ($first2 != '/*' || $first3 == '/*!')) {
      $out .= ' ' . trim($line);
      // If a line ends in ; or */ it is a sql command.
      if (substr($out, strlen($out) - 1, 1) == ';') {
        return trim($out);
      }
    }
  }
  return trim($out);
}

/**
 * Get mysqli connection.
 */
function _fast_get_mysqli_connection() {
  if (!function_exists('mysqli_init') && !extension_loaded('mysqli')) {
    throw new \Exception('Cannot connect to the database because the MySQLi extension is missing.');
  }

  $db = Database::getConnection();
  $options = $db->getConnectionOptions();

  $connection = new \mysqli(
    $options['host'],
    $options['username'],
    $options['password'],
    $options['database'],
    $options['port'],
    $options['socket'] ?? '',
  );

  // Throw an error on fail.
  if ($connection->connect_errno) {
    throw new \Exception("Failed to connect to MySQL server.");
  }
  // Ensure, that the character set is utf8mb4.
  if (!$connection->set_charset('utf8mb4')) {
    throw new \Exception('UTF8 is not supported by the MySQL server.');
  }

  return $connection;
}

/**
 * Drop all tables in the default database.
 */
function _fast_drop_all_tables() {
  $connection = Database::getConnection();
  $schema = $connection->schema();

  // Get the list of all tables.
  $tables = $connection->query("SHOW TABLES")->fetchCol();

  foreach ($tables as $table) {
    try {
      $schema->dropTable($table);
    } catch (SchemaObjectExistsException $e) {
      // Handle the exception if the table does not exist.
      \Drupal::logger('fast')->notice('Table %table does not exist or could not be dropped.', ['%table' => $table]);
    }
  }
}

/**
 * Returns a list of tables in the active database.
 *
 * Only returns tables whose prefix matches the configured one (or ones, if
 * there are multiple).
 *
 * @see demo_enum_tables()
 */
function _fast_list_tables() {
  // Get the database connection.
  $connection = Database::getConnection();

  // Get the database prefix.
  $db_prefix = $connection->getPrefix();

  if (is_array($db_prefix)) {
    // Create a regular expression for table prefix matching.
    $rx = '/^' . implode('|', array_filter($db_prefix)) . '/';
  }
  else {
    if ($db_prefix != '') {
      $rx = '/^' . $db_prefix . '/';
    }
  }

  $tables = _fast_find_tables('%');

  foreach ($tables as $table) {
    if (is_array($db_prefix)) {
      // Check if table name matches a configured prefix.
      if (preg_match($rx, $table, $matches)) {
        $table_prefix = $matches[0];
        $plain_table = substr($table, strlen($table_prefix));
        if ($db_prefix[$plain_table] == $table_prefix || $db_prefix['default'] == $table_prefix) {
          $tables[] = $table;
        }
      }
    }
    else {
      if ($db_prefix != '') {
        if (preg_match($rx, $table)) {
          $tables[] = $table;
        }
      }
      else {
        $tables[] = $table;
      }
    }
  }

  return $tables;
}

/**
 * Find all tables in the database.
 *
 * @param string $like
 *   The pattern to match tables against.
 *
 * @return array
 *   An array of table names.
 */
function _fast_find_tables($like = '%') {
  $connection = Database::getConnection();
  $query = $connection->query("SHOW TABLES LIKE :pattern", [':pattern' => $like]);
  return $query->fetchCol();
}
